@extends('layouts.UserLayout')

@push('styles')
<style>
/* ================= GLOBAL STYLES ================= */
* {
    margin: 0;
    padding: 0;
    box-sizing: border-box;
}

body {
    background: #f0f2f5;
    font-family: 'Inter', -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
    color: #1e293b;
    line-height: 1.5;
}

/* ================= DESKTOP REPORT STYLES ================= */
.desktop-report {
    max-width: 1280px;
    margin: 40px auto;
    padding: 0 24px;
}

/* Action Buttons */
.action-buttons {
    display: flex;
    justify-content: flex-end;
    gap: 16px;
    margin-bottom: 24px;
}

.action-btn {
    padding: 12px 28px;
    border-radius: 40px;
    font-weight: 600;
    font-size: 0.95rem;
    display: inline-flex;
    align-items: center;
    gap: 10px;
    text-decoration: none;
    transition: all 0.2s;
    border: none;
    cursor: pointer;
    background: white;
    color: #1e293b;
    border: 1px solid #cbd5e1;
    box-shadow: 0 2px 6px rgba(0,0,0,0.02);
}

.action-btn.excel {
    background: #0f6e4a;
    color: white;
    border: none;
}

.action-btn.pdf {
    background: #b22222;
    color: white;
    border: none;
}

.action-btn.print {
    background: #2563eb;
    color: white;
    border: none;
}

.action-btn:hover {
    transform: translateY(-2px);
    box-shadow: 0 8px 16px rgba(0,0,0,0.08);
}

/* Report Container */
.report-container {
    background: white;
    border-radius: 24px;
    box-shadow: 0 12px 30px rgba(0, 0, 0, 0.08);
    overflow: hidden;
    border: 1px solid #e9eef2;
}

/* Report Header - Enhanced for Official Report */
.report-header {
    background: linear-gradient(145deg, #0a1c2f 0%, #0e2a40 100%);
    color: white;
    padding: 36px 40px;
    position: relative;
    border-bottom: 4px solid #c7a03f;
}

.report-header::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    height: 6px;
    background: linear-gradient(90deg, #c7a03f, #e5c07b, #c7a03f);
}

.report-title-section {
    display: flex;
    justify-content: space-between;
    align-items: flex-start;
    margin-bottom: 28px;
}

.report-main-title h1 {
    font-size: 2.2rem;
    font-weight: 700;
    margin: 0 0 8px 0;
    letter-spacing: -0.01em;
    text-transform: uppercase;
    color: white;
}

.report-main-title .hotel-name {
    font-size: 1.2rem;
    opacity: 0.9;
    display: flex;
    align-items: center;
    gap: 8px;
    color: #e5e7eb;
}

.report-badge {
    background: rgba(199, 160, 63, 0.2);
    padding: 8px 20px;
    border-radius: 40px;
    font-size: 0.9rem;
    font-weight: 600;
    display: flex;
    align-items: center;
    gap: 8px;
    backdrop-filter: blur(4px);
    border: 1px solid rgba(199, 160, 63, 0.4);
    color: #f3e8c0;
}

.report-id {
    font-size: 0.9rem;
    color: #b8d1e0;
    margin-top: 8px;
    font-family: monospace;
    letter-spacing: 0.5px;
}

/* Shift Info Grid */
.shift-info-grid {
    display: grid;
    grid-template-columns: repeat(4, 1fr);
    gap: 24px;
    margin-top: 24px;
}

.info-card {
    background: rgba(255, 255, 255, 0.08);
    border-radius: 16px;
    padding: 18px 22px;
    backdrop-filter: blur(4px);
    border: 1px solid rgba(255, 255, 255, 0.1);
}

.info-label {
    font-size: 0.8rem;
    text-transform: uppercase;
    letter-spacing: 0.5px;
    opacity: 0.7;
    margin-bottom: 8px;
    display: flex;
    align-items: center;
    gap: 6px;
}

.info-value {
    font-size: 1.4rem;
    font-weight: 700;
    line-height: 1.2;
}

.info-sub {
    font-size: 0.85rem;
    opacity: 0.8;
    margin-top: 6px;
}

/* Executive Summary */
.executive-summary {
    display: grid;
    grid-template-columns: repeat(4, 1fr);
    gap: 24px;
    padding: 32px 40px;
    background: #f8fafd;
    border-bottom: 1px solid #e2e8f0;
}

.stat-item {
    display: flex;
    align-items: center;
    gap: 18px;
}

.stat-icon {
    width: 56px;
    height: 56px;
    background: white;
    border-radius: 18px;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 1.6rem;
    color: #0a1c2f;
    box-shadow: 0 6px 14px rgba(0,0,0,0.02);
    border: 1px solid #e2e8f0;
}

.stat-details h4 {
    font-size: 0.85rem;
    font-weight: 600;
    color: #5e6f7e;
    margin-bottom: 6px;
    text-transform: uppercase;
    letter-spacing: 0.3px;
}

.stat-number {
    font-size: 1.9rem;
    font-weight: 800;
    color: #0a1c2f;
    line-height: 1;
    margin-bottom: 4px;
}

.stat-label {
    font-size: 0.8rem;
    color: #5e6f7e;
}

/* Section Styles */
.report-section {
    padding: 36px 40px;
    border-bottom: 1px solid #e9eef2;
}

.report-section:last-child {
    border-bottom: none;
}

.section-header {
    display: flex;
    align-items: center;
    gap: 14px;
    margin-bottom: 28px;
}

.section-icon {
    width: 48px;
    height: 48px;
    background: linear-gradient(145deg, #0a1c2f, #0e2a40);
    border-radius: 14px;
    display: flex;
    align-items: center;
    justify-content: center;
    color: white;
    font-size: 1.4rem;
}

.section-title {
    font-size: 1.5rem;
    font-weight: 700;
    color: #0a1c2f;
    margin: 0;
}

.section-sub {
    font-size: 0.9rem;
    color: #5e6f7e;
    margin-top: 4px;
}

/* Guest Feedback Cards */
.guest-feedback-grid {
    display: grid;
    grid-template-columns: repeat(3, 1fr);
    gap: 24px;
}

.feedback-card {
    background: #f9fcff;
    border-radius: 20px;
    padding: 24px;
    border: 1px solid #e2eaf0;
    transition: all 0.2s;
    box-shadow: 0 2px 8px rgba(0,0,0,0.02);
}

.feedback-card:hover {
    border-color: #b6d0e0;
    box-shadow: 0 8px 20px rgba(0,50,80,0.04);
}

.feedback-header {
    display: flex;
    justify-content: space-between;
    margin-bottom: 16px;
    padding-bottom: 12px;
    border-bottom: 1px dashed #d0dfe8;
}

.guest-name {
    font-weight: 700;
    font-size: 1.1rem;
    color: #0a1c2f;
}

.room-tag {
    background: #e5edf2;
    padding: 4px 12px;
    border-radius: 30px;
    font-size: 0.8rem;
    font-weight: 600;
    color: #1e4a6b;
}

.feedback-text {
    color: #2c3e50;
    line-height: 1.7;
    font-style: italic;
    margin-bottom: 16px;
    padding-left: 12px;
    border-left: 4px solid #c7a03f;
    font-size: 0.95rem;
}

/* Inspection Tables - Enhanced */
.inspection-table {
    width: 100%;
    border-collapse: collapse;
    margin-top: 12px;
    border: 1px solid #e2e8f0;
    border-radius: 12px;
    overflow: hidden;
}

.inspection-table th {
    text-align: left;
    padding: 16px 14px;
    background: #f1f6fa;
    color: #1e405b;
    font-weight: 600;
    font-size: 0.85rem;
    text-transform: uppercase;
    letter-spacing: 0.3px;
    border-bottom: 2px solid #cbd5e1;
}

.inspection-table td {
    padding: 14px 14px;
    border-bottom: 1px solid #ecf1f5;
    color: #2c3e50;
}

.inspection-table tr:last-child td {
    border-bottom: none;
}

.status-ok {
    background: #e2f3e2;
    color: #1a6d3a;
    padding: 4px 14px;
    border-radius: 30px;
    font-size: 0.8rem;
    font-weight: 600;
    display: inline-block;
}

.status-issue {
    background: #ffebeb;
    color: #b33a3a;
    padding: 4px 14px;
    border-radius: 30px;
    font-size: 0.8rem;
    font-weight: 600;
    display: inline-block;
}

/* Area Cards */
.area-grid {
    display: grid;
    grid-template-columns: repeat(2, 1fr);
    gap: 24px;
}

.area-card {
    background: #ffffff;
    border-radius: 18px;
    padding: 24px;
    border: 1px solid #e4ecf2;
    box-shadow: 0 4px 12px rgba(0,20,40,0.02);
}

.area-name {
    font-weight: 700;
    font-size: 1.1rem;
    color: #0a1c2f;
    margin-bottom: 16px;
    display: flex;
    align-items: center;
    gap: 10px;
}

.area-name i {
    color: #3a6c8c;
}

.condition-list {
    list-style: none;
    padding: 0;
    margin: 0;
}

.condition-list li {
    display: flex;
    align-items: flex-start;
    gap: 12px;
    padding: 10px 0;
    border-bottom: 1px solid #f0f5f8;
}

.condition-list li:last-child {
    border-bottom: none;
}

/* Employee Recognition */
.recognition-card {
    background: linear-gradient(145deg, #fdf9e6, #fffef7);
    border-radius: 20px;
    padding: 28px;
    border-left: 6px solid #c7a03f;
    margin-top: 12px;
    box-shadow: 0 4px 12px rgba(199,160,63,0.08);
}

.recognition-name {
    font-size: 1.3rem;
    font-weight: 700;
    color: #854d0e;
    margin-bottom: 8px;
}

.recognition-dept {
    font-size: 0.95rem;
    color: #b45309;
    margin-bottom: 16px;
    display: inline-block;
    background: #fde68a;
    padding: 4px 16px;
    border-radius: 30px;
}

.recognition-story {
    font-size: 1rem;
    line-height: 1.7;
    color: #434343;
}

/* Signature Section */
.signature-section {
    display: flex;
    justify-content: space-between;
    align-items: flex-end;
    margin-top: 40px;
    padding-top: 32px;
    border-top: 2px solid #dce5ea;
}

.signature-box {
    flex: 1;
}

.signature-line {
    display: flex;
    align-items: center;
    gap: 20px;
    margin-bottom: 24px;
}

.signature-label {
    font-size: 0.85rem;
    color: #5e6f7e;
    text-transform: uppercase;
    letter-spacing: 1px;
    min-width: 120px;
}

.signature-placeholder {
    border-bottom: 2px solid #b8cad2;
    min-width: 260px;
    padding-bottom: 8px;
    font-family: 'Caveat', cursive;
    font-size: 1.2rem;
    color: #1e405b;
}

.signature-date {
    color: #3a5a6a;
    font-size: 0.9rem;
}

.report-footer {
    background: #f4f8fc;
    padding: 24px 40px;
    display: flex;
    justify-content: space-between;
    align-items: center;
    font-size: 0.85rem;
    color: #4a6676;
    border-top: 1px solid #dce5ea;
}

/* ================= MOBILE REPORT STYLES ================= */
.mobile-report {
    display: none;
    padding: 16px;
    background: #f8fafc;
    min-height: 100vh;
}

.mobile-report-container {
    background: white;
    border-radius: 20px;
    overflow: hidden;
    box-shadow: 0 6px 18px rgba(0,0,0,0.04);
}

.mobile-header {
    background: linear-gradient(145deg, #0a1c2f, #0e2a40);
    color: white;
    padding: 24px 20px;
    border-bottom: 4px solid #c7a03f;
}

.mobile-header h1 {
    font-size: 1.6rem;
    font-weight: 700;
    margin-bottom: 6px;
    text-transform: uppercase;
}

.mobile-action-buttons {
    display: flex;
    gap: 10px;
    margin: 20px 0 10px;
}

.mobile-action-btn {
    flex: 1;
    padding: 12px;
    border-radius: 40px;
    font-size: 0.85rem;
    font-weight: 600;
    display: flex;
    align-items: center;
    justify-content: center;
    gap: 8px;
    text-decoration: none;
    color: white;
}

.mobile-action-btn.excel { background: #0f6e4a; }
.mobile-action-btn.pdf { background: #b22222; }
.mobile-action-btn.print { background: #2563eb; }

.mobile-stats-grid {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 12px;
    margin-top: 20px;
}

.mobile-stat-card {
    background: rgba(255,255,255,0.1);
    border-radius: 16px;
    padding: 16px 14px;
    backdrop-filter: blur(4px);
}

.mobile-stat-label {
    font-size: 0.75rem;
    opacity: 0.8;
    margin-bottom: 6px;
}

.mobile-stat-value {
    font-size: 1.5rem;
    font-weight: 700;
}

.mobile-section {
    padding: 24px 20px;
    border-bottom: 1px solid #e9eef2;
}

.mobile-section-title {
    font-size: 1.2rem;
    font-weight: 700;
    color: #0a1c2f;
    margin-bottom: 20px;
    display: flex;
    align-items: center;
    gap: 10px;
}

.mobile-feedback-card {
    background: #f9fcff;
    border-radius: 18px;
    padding: 20px;
    margin-bottom: 16px;
    border: 1px solid #e2eaf0;
}

.mobile-area-grid {
    display: flex;
    flex-direction: column;
    gap: 16px;
}

.mobile-area-item {
    background: #f9fcff;
    border-radius: 16px;
    padding: 18px;
    border: 1px solid #e2eaf0;
}

.mobile-signature {
    display: flex;
    flex-direction: column;
    gap: 16px;
    margin-top: 20px;
}

.mobile-signature-row {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 12px 0;
    border-bottom: 1px solid #ecf1f5;
}

/* ================= RESPONSIVE ================= */
@media (max-width: 1023px) {
    .desktop-report { display: none; }
    .mobile-report { display: block; }
}

@media (min-width: 1024px) {
    .desktop-report { display: block; }
    .mobile-report { display: none; }
}

/* ================= PRINT STYLES ================= */
@media print {
    body { background: white; }
    .desktop-report { display: block; margin: 0; padding: 0; }
    .report-container { box-shadow: none; border: 1px solid #000; }
    .no-print, .action-buttons, .mobile-action-buttons, 
    .btn-back, .btn-print, .action-btn, .mobile-action-btn {
        display: none !important;
    }
    .report-header { background: #0a1c2f !important; -webkit-print-color-adjust: exact; print-color-adjust: exact; }
    .status-ok, .status-issue { -webkit-print-color-adjust: exact; print-color-adjust: exact; }
}
</style>
@endpush

{{-- ================= DESKTOP REPORT SECTION ================= --}}
@section('content')
@php
    // Default values
    $checklist = $checklist ?? null;
    $submissions = $submissions ?? [];
    $sections = $sections ?? collect();
    $allQuestionAnswers = $allQuestionAnswers ?? [];
    $allSubQuestionAnswers = $allSubQuestionAnswers ?? [];
    $user = auth()->user();
    $today = now();
    
    // Ambil data MOD session jika ada
    $modSession = \App\Models\ModSession::where('user_id', $user->id)
        ->where('event_date', $today->toDateString())
        ->where('hotel', $checklist->hotel ?? '')
        ->first();
    
    // Hitung total rooms inspected
    $totalRoomsInspected = 0;
    $guestFeedbacks = [];
    $inspectedRooms = [];
    
    if ($checklist && $checklist->mod_type === 'room_inspection' && !empty($submissions)) {
        $totalRoomsInspected = count($submissions);
        // Ambil maksimal 3 guest feedback untuk laporan
        $feedbackCounter = 0;
        foreach ($submissions as $roomNumber => $submission) {
            if ($feedbackCounter < 3 && $submission->overall_comment) {
                $guestFeedbacks[] = [
                    'guest_name' => 'Guest Room ' . $roomNumber,
                    'room' => $roomNumber,
                    'arrival' => $submission->created_at->format('d M'),
                    'departure' => $submission->created_at->copy()->addDays(1)->format('d M'),
                    'feedback' => $submission->overall_comment
                ];
                $feedbackCounter++;
            }
            $inspectedRooms[] = $roomNumber;
        }
    }
    
    // Data simulasi (akan diganti dengan data real dari DB nanti)
    $occupancyRate = $modSession ? rand(65, 95) : 78;
    $arrivals = rand(12, 28);
    $departures = rand(10, 24);
    $guestsInHouse = rand(120, 210);
    $vipArrivals = rand(2, 6);
    
    $publicAreas = [
        ['area' => 'Lobby & Reception', 'status' => 'OK', 'notes' => 'Clean, well-lit, staff presentable'],
        ['area' => 'Lobby Toilets', 'status' => 'OK', 'notes' => 'Fresh scent, fully stocked'],
        ['area' => 'Entrance & Driveway', 'status' => 'OK', 'notes' => 'Flags in good condition'],
        ['area' => 'Guest Lifts', 'status' => 'OK', 'notes' => 'Smooth operation, clean interiors'],
        ['area' => 'Ballroom Pre-function', 'status' => 'Issue', 'notes' => 'One fused bulb near door #3'],
        ['area' => 'Digital Signages', 'status' => 'OK', 'notes' => 'Current events displayed']
    ];
@endphp

@if(!$checklist)
    <div class="desktop-report">
        <div style="background: white; border-radius: 24px; padding: 60px; text-align: center;">
            <i class="fas fa-file-excel" style="font-size: 4rem; color: #a0b8c5; margin-bottom: 20px;"></i>
            <h2 style="color: #1e405b;">Report Tidak Tersedia</h2>
            <p style="color: #5e6f7e; margin-top: 12px;">Checklist tidak ditemukan atau belum ada inspeksi hari ini.</p>
        </div>
    </div>
@else
<div class="desktop-report">
    

    <!-- Main Report Container -->
    <div class="report-container">
        <!-- HEADER - Official MOD Report -->
        <div class="report-header">
            <div class="report-title-section">
                <div class="report-main-title">
                    <h1>MANAGER ON DUTY REPORT</h1>
                    <div class="hotel-name">
                        <i class="fas fa-hotel"></i> {{ $checklist->hotel }}
                    </div>
                    <div class="report-id">
                        Report No: MOD/{{ $today->format('Ymd') }}/{{ str_pad($checklist->id, 4, '0', STR_PAD_LEFT) }}
                    </div>
                </div>
                <div class="report-badge">
                    <i class="fas fa-shield-alt"></i> OFFICIAL - STANDARD OPERATION
                </div>
            </div>

            <!-- Shift Information -->
            <div class="shift-info-grid">
                <div class="info-card">
                    <div class="info-label"><i class="fas fa-user-tie"></i> MOD ON DUTY</div>
                    <div class="info-value">{{ $user->name }}</div>
                    <div class="info-sub">{{ $user->position ?? 'Manager on Duty' }}</div>
                </div>
                <div class="info-card">
                    <div class="info-label"><i class="fas fa-calendar-alt"></i> REPORT DATE</div>
                    <div class="info-value">{{ $today->format('d M Y') }}</div>
                    <div class="info-sub">{{ $today->format('l') }}</div>
                </div>
                <div class="info-card">
                    <div class="info-label"><i class="fas fa-clock"></i> SHIFT TIME</div>
                    <div class="info-value">09:00 - 17:00</div>
                    <div class="info-sub">({{ $modSession ? $modSession->room_inspection_count ?? 0 : 0 }} inspections)</div>
                </div>
                <div class="info-card">
                    <div class="info-label"><i class="fas fa-cloud-sun"></i> WEATHER</div>
                    <div class="info-value">{{ $today->format('M') }} • Partly Cloudy</div>
                    <div class="info-sub">24°C / 31°C</div>
                </div>
            </div>
        </div>

        <!-- EXECUTIVE SUMMARY -->
        <div class="executive-summary">
            <div class="stat-item">
                <div class="stat-icon"><i class="fas fa-door-open"></i></div>
                <div class="stat-details">
                    <h4>OCCUPANCY</h4>
                    <div class="stat-number">{{ $occupancyRate }}%</div>
                    <div class="stat-label">{{ $guestsInHouse }} guests in-house</div>
                </div>
            </div>
            <div class="stat-item">
                <div class="stat-icon"><i class="fas fa-calendar-check"></i></div>
                <div class="stat-details">
                    <h4>ARR / DEP</h4>
                    <div class="stat-number">{{ $arrivals }} / {{ $departures }}</div>
                    <div class="stat-label">Expected arrivals/departures</div>
                </div>
            </div>
            <div class="stat-item">
                <div class="stat-icon"><i class="fas fa-clipboard-check"></i></div>
                <div class="stat-details">
                    <h4>ROOMS INSPECTED</h4>
                    <div class="stat-number">{{ $totalRoomsInspected }}</div>
                    <div class="stat-label">{{ $checklist->mod_type === 'room_inspection' ? 'Guest rooms' : 'General operation' }}</div>
                </div>
            </div>
            <div class="stat-item">
                <div class="stat-icon"><i class="fas fa-crown"></i></div>
                <div class="stat-details">
                    <h4>VIP ARRIVALS</h4>
                    <div class="stat-number">{{ $vipArrivals }}</div>
                    <div class="stat-label">Elite/Ambassador</div>
                </div>
            </div>
        </div>

        <!-- GUEST FEEDBACK SECTION -->
        @if($checklist->mod_type === 'room_inspection' && !empty($guestFeedbacks))
        <div class="report-section">
            <div class="section-header">
                <div class="section-icon"><i class="fas fa-star"></i></div>
                <div>
                    <h2 class="section-title">Guest Feedback Summary</h2>
                    <div class="section-sub">Feedback from departing guests - {{ count($guestFeedbacks) }} entries</div>
                </div>
            </div>
            <div class="guest-feedback-grid">
                @foreach($guestFeedbacks as $feedback)
                <div class="feedback-card">
                    <div class="feedback-header">
                        <span class="guest-name">{{ $feedback['guest_name'] }}</span>
                        <span class="room-tag">Room #{{ $feedback['room'] }}</span>
                    </div>
                    <div style="display: flex; gap: 12px; margin-bottom: 16px; font-size: 0.8rem; color: #5e6f7e;">
                        <span><i class="far fa-calendar"></i> Arr: {{ $feedback['arrival'] }}</span>
                        <span><i class="far fa-calendar-check"></i> Dep: {{ $feedback['departure'] }}</span>
                    </div>
                    <div class="feedback-text">
                        “{{ $feedback['feedback'] }}”
                    </div>
                    <div style="display: flex; justify-content: flex-end;">
                        <span style="background: #eff6ff; padding: 4px 12px; border-radius: 30px; font-size: 0.75rem; color: #2563eb;">
                            <i class="fas fa-check-circle"></i> Recorded by MOD
                        </span>
                    </div>
                </div>
                @endforeach
            </div>
        </div>
        @endif

        <!-- PUBLIC AREAS INSPECTION -->
        <div class="report-section">
            <div class="section-header">
                <div class="section-icon"><i class="fas fa-building"></i></div>
                <div>
                    <h2 class="section-title">Public Areas & Facilities</h2>
                    <div class="section-sub">Front of house inspection summary</div>
                </div>
            </div>

            <table class="inspection-table">
                <thead>
                    <tr>
                        <th>Area / Location</th>
                        <th>Status</th>
                        <th>Remarks / Action Taken</th>
                    </tr>
                </thead>
                <tbody>
                    @foreach($publicAreas as $area)
                    <tr>
                        <td style="font-weight: 500;">{{ $area['area'] }}</td>
                        <td>
                            <span class="{{ $area['status'] === 'OK' ? 'status-ok' : 'status-issue' }}">
                                {{ $area['status'] }}
                            </span>
                        </td>
                        <td style="color: #3a5a6a;">{{ $area['notes'] }}</td>
                    </tr>
                    @endforeach
                </tbody>
            </table>

            <!-- Photo Attachment Note -->
            <div style="margin-top: 24px; padding: 16px; background: #f6fafd; border-radius: 14px; display: flex; align-items: center; gap: 12px;">
                <i class="fas fa-camera" style="color: #3a6c8c; font-size: 1.2rem;"></i>
                <span style="color: #2d4a5e;">Photo documentation attached for ballroom lighting issue. Maintenance notified.</span>
            </div>
        </div>

        <!-- F&B OUTLETS -->
        <div class="report-section">
            <div class="section-header">
                <div class="section-icon"><i class="fas fa-utensils"></i></div>
                <div>
                    <h2 class="section-title">Food & Beverage Outlets</h2>
                    <div class="section-sub">Dining experience & kitchen inspection</div>
                </div>
            </div>

            <div class="area-grid">
                <div class="area-card">
                    <div class="area-name"><i class="fas fa-coffee"></i> Just Grill / Just Wok</div>
                    <ul class="condition-list">
                        <li><i class="fas fa-check-circle" style="color: #1a9c50;"></i> Entrance clean & tidy</li>
                        <li><i class="fas fa-check-circle" style="color: #1a9c50;"></i> Furniture in good condition</li>
                        <li><i class="fas fa-check-circle" style="color: #1a9c50;"></i> Music system playing</li>
                        <li><i class="fas fa-exclamation-triangle" style="color: #d97706;"></i> Menu covers slightly worn (to be replaced)</li>
                        <li><i class="fas fa-check-circle" style="color: #1a9c50;"></i> Toilets clean & stocked</li>
                    </ul>
                </div>
                <div class="area-card">
                    <div class="area-name"><i class="fas fa-glass-cheers"></i> SANGS Bar & Pool</div>
                    <ul class="condition-list">
                        <li><i class="fas fa-check-circle" style="color: #1a9c50;"></i> Adequate colleagues</li>
                        <li><i class="fas fa-check-circle" style="color: #1a9c50;"></i> Deck & beach area clean</li>
                        <li><i class="fas fa-check-circle" style="color: #1a9c50;"></i> Cocktail of the day available</li>
                        <li><i class="fas fa-check-circle" style="color: #1a9c50;"></i> Pool clean & vacuumed</li>
                        <li><i class="fas fa-check-circle" style="color: #1a9c50;"></i> Sun loungers in good condition</li>
                    </ul>
                </div>
            </div>
        </div>

        <!-- RECREATIONAL FACILITIES -->
        <div class="report-section">
            <div class="section-header">
                <div class="section-icon"><i class="fas fa-swimmer"></i></div>
                <div>
                    <h2 class="section-title">Spa, Gym & Pool</h2>
                    <div class="section-sub">Wellness facilities status</div>
                </div>
            </div>

            <div class="area-grid">
                <div class="area-card">
                    <div class="area-name"><i class="fas fa-spa"></i> ELENA Spa</div>
                    <ul class="condition-list">
                        <li><i class="fas fa-check-circle" style="color: #1a9c50;"></i> Reception manned</li>
                        <li><i class="fas fa-check-circle" style="color: #1a9c50;"></i> Ambience & music appropriate</li>
                        <li><i class="fas fa-check-circle" style="color: #1a9c50;"></i> Relaxation area tidy</li>
                        <li><i class="fas fa-check-circle" style="color: #1a9c50;"></i> Water pond clean, lights working</li>
                    </ul>
                </div>
                <div class="area-card">
                    <div class="area-name"><i class="fas fa-dumbbell"></i> Fitness Centre</div>
                    <ul class="condition-list">
                        <li><i class="fas fa-check-circle" style="color: #1a9c50;"></i> Towels available, fresh</li>
                        <li><i class="fas fa-check-circle" style="color: #1a9c50;"></i> Equipment in working order</li>
                        <li><i class="fas fa-check-circle" style="color: #1a9c50;"></i> Locker rooms clean, odor-free</li>
                        <li><i class="fas fa-check-circle" style="color: #1a9c50;"></i> Weighing scale functional</li>
                    </ul>
                </div>
            </div>
        </div>

        <!-- GUEST ROOM INSPECTION -->
        @if($checklist->mod_type === 'room_inspection' && !empty($inspectedRooms))
        <div class="report-section">
            <div class="section-header">
                <div class="section-icon"><i class="fas fa-bed"></i></div>
                <div>
                    <h2 class="section-title">Guest Room Inspection</h2>
                    <div class="section-sub">{{ count($inspectedRooms) }} rooms inspected, including VIP arrivals</div>
                </div>
            </div>

            <div style="display: flex; flex-wrap: wrap; gap: 12px; margin-bottom: 24px;">
                @foreach($inspectedRooms as $room)
                <span style="background: #eef2f6; padding: 8px 18px; border-radius: 30px; font-weight: 600; font-size: 0.9rem;">
                    <i class="fas fa-door-closed"></i> Room {{ $room }}
                </span>
                @endforeach
            </div>

            <table class="inspection-table">
                <thead>
                    <tr>
                        <th>Room #</th>
                        <th>AC / Lights</th>
                        <th>Cleanliness</th>
                        <th>Pool (Lagoon)</th>
                        <th>Overall Comment</th>
                    </tr>
                </thead>
                <tbody>
                    @foreach($submissions as $roomNumber => $submission)
                    <tr>
                        <td><strong>{{ $roomNumber }}</strong></td>
                        <td><span class="status-ok">OK</span></td>
                        <td><span class="status-ok">OK</span></td>
                        <td><span class="status-ok">Clean</span></td>
                        <td style="max-width: 250px;">{{ $submission->overall_comment ?? 'No comment' }}</td>
                    </tr>
                    @endforeach
                </tbody>
            </table>
        </div>
        @endif

        <!-- BACK OF HOUSE -->
        <div class="report-section">
            <div class="section-header">
                <div class="section-icon"><i class="fas fa-hard-hat"></i></div>
                <div>
                    <h2 class="section-title">Back of House & Heart of House</h2>
                    <div class="section-sub">Staff facilities, kitchen, engineering</div>
                </div>
            </div>

            <div class="area-grid">
                <div class="area-card">
                    <div class="area-name"><i class="fas fa-utensil-spoon"></i> Staff Café / Associate Dining</div>
                    <ul class="condition-list">
                        <li><i class="fas fa-check-circle" style="color: #1a9c50;"></i> Food quality: Good (4/5)</li>
                        <li><i class="fas fa-check-circle" style="color: #1a9c50;"></i> Variety: Satisfactory</li>
                        <li><i class="fas fa-check-circle" style="color: #1a9c50;"></i> Cleanliness: All light bulbs functional</li>
                        <li><i class="fas fa-check-circle" style="color: #1a9c50;"></i> Floor & furniture clean</li>
                    </ul>
                </div>
                <div class="area-card">
                    <div class="area-name"><i class="fas fa-lock"></i> Associate Locker Rooms</div>
                    <ul class="condition-list">
                        <li><i class="fas fa-check-circle" style="color: #1a9c50;"></i> Male: Washroom clean, no odor</li>
                        <li><i class="fas fa-check-circle" style="color: #1a9c50;"></i> Female: Lockers tidy</li>
                        <li><i class="fas fa-check-circle" style="color: #1a9c50;"></i> Lighting: All working</li>
                        <li><i class="fas fa-check-circle" style="color: #1a9c50;"></i> Ventilation: Good</li>
                    </ul>
                </div>
            </div>
            <div style="margin-top: 20px; background: #f6fafd; border-radius: 14px; padding: 20px;">
                <div style="display: flex; gap: 24px; flex-wrap: wrap;">
                    <span><i class="fas fa-trash-alt" style="color: #4b6a7c;"></i> Incinerator room: Clear</span>
                    <span><i class="fas fa-tools" style="color: #4b6a7c;"></i> Engineering building: Clean surroundings</span>
                    <span><i class="fas fa-badminton" style="color: #4b6a7c;"></i> Badminton court: Free of rubbish</span>
                    <span><i class="fas fa-tennis-ball" style="color: #4b6a7c;"></i> Tennis court: Properly locked</span>
                </div>
            </div>
        </div>

        <!-- EMPLOYEE RECOGNITION -->
        <div class="report-section">
            <div class="section-header">
                <div class="section-icon"><i class="fas fa-award"></i></div>
                <div>
                    <h2 class="section-title">Employee Recognition</h2>
                    <div class="section-sub">Star of the shift - Going above and beyond</div>
                </div>
            </div>
            <div class="recognition-card">
                <div class="recognition-name">Putu Ayu - Front Office</div>
                <div class="recognition-dept">Guest Service Agent</div>
                <div class="recognition-story">
                    "Assisted a VIP family with early check-in at 08:30, arranged connecting rooms and provided welcome amenities personally. Guest left a glowing verbal feedback at checkout. Exemplary service attitude."
                </div>
                <div style="margin-top: 20px; display: flex; justify-content: space-between; align-items: center;">
                    <span style="color: #92400e;"><i class="fas fa-heart"></i> Recognised by MOD</span>
                    <span style="background: white; padding: 4px 16px; border-radius: 30px; font-size: 0.8rem; border: 1px solid #c7a03f; color: #854d0e; font-weight: 600;">
                        🌟 Star of the shift
                    </span>
                </div>
            </div>
        </div>

        <!-- SIGNATURE & HANDOVER -->
        <div style="padding: 36px 40px;">
            <div class="signature-section">
                <div class="signature-box">
                    <div class="signature-line">
                        <span class="signature-label">Prepared by:</span>
                        <span class="signature-placeholder">{{ $user->name }}</span>
                    </div>
                    <div class="signature-line">
                        <span class="signature-label">Reviewed by:</span>
                        <span class="signature-placeholder"></span>
                    </div>
                    <div style="display: flex; gap: 40px; margin-top: 20px;">
                        <div><span style="color: #5e6f7e;">Date:</span> {{ $today->format('d M Y') }}</div>
                        <div><span style="color: #5e6f7e;">Time:</span> {{ $today->format('h:i A') }}</div>
                    </div>
                </div>
                <div style="text-align: right;">
                    <div style="background: #eaf0f5; padding: 20px 28px; border-radius: 16px;">
                        <div style="font-weight: 700; color: #1e405b; margin-bottom: 10px; font-size: 1.1rem;">
                            <i class="fas fa-exchange-alt"></i> MOD Shift Handover
                        </div>
                        <div style="font-size: 0.95rem; color: #2d4a5e;">iPad returned • Report distributed to EXCOM</div>
                        <div style="font-size: 0.85rem; color: #4f6f7f; margin-top: 12px;">
                            Next MOD: Duty Manager (09:00)
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- FOOTER -->
        <div class="report-footer">
            <div><i class="fas fa-check-circle" style="color: #1a9c50;"></i> This is an official MOD report generated from the checklist system</div>
            <div style="font-family: monospace;">MOD/{{ $today->format('Ymd') }}/{{ str_pad($checklist->id, 4, '0', STR_PAD_LEFT) }} · CONFIDENTIAL</div>
        </div>
    </div>
</div>
@endif
@endsection

{{-- ================= MOBILE REPORT SECTION ================= --}}
@section('mobile-content')
@php
    // Reuse variables from desktop
    $checklist = $checklist ?? null;
    $submissions = $submissions ?? [];
    $user = auth()->user();
    $today = now();
@endphp

@if(!$checklist)
    <div class="mobile-report">
        <div style="background: white; border-radius: 20px; padding: 40px 20px; text-align: center;">
            <i class="fas fa-file-excel" style="font-size: 3rem; color: #a0b8c5;"></i>
            <h3 style="margin-top: 16px;">Report Tidak Tersedia</h3>
        </div>
    </div>
@else
<div class="mobile-report">
    <div class="mobile-report-container">
        <!-- Mobile Header -->
        <div class="mobile-header">
            <h1>MOD REPORT</h1>
            <div style="display: flex; align-items: center; gap: 8px; margin-bottom: 12px;">
                <i class="fas fa-hotel"></i>
                <span style="font-size: 1.1rem;">{{ $checklist->hotel }}</span>
            </div>
            <div style="font-size: 0.9rem; opacity: 0.9;">
                {{ $user->name }} · {{ $today->format('d M Y, h:i A') }}
            </div>
            
            
            
            <div class="mobile-stats-grid">
                <div class="mobile-stat-card">
                    <div class="mobile-stat-label">OCC</div>
                    <div class="mobile-stat-value">{{ $occupancyRate ?? 78 }}%</div>
                </div>
                <div class="mobile-stat-card">
                    <div class="mobile-stat-label">ARR/DEP</div>
                    <div class="mobile-stat-value">{{ $arrivals ?? 18 }}/{{ $departures ?? 14 }}</div>
                </div>
                <div class="mobile-stat-card">
                    <div class="mobile-stat-label">ROOMS</div>
                    <div class="mobile-stat-value">{{ count($submissions) }}</div>
                </div>
                <div class="mobile-stat-card">
                    <div class="mobile-stat-label">VIP</div>
                    <div class="mobile-stat-value">{{ $vipArrivals ?? 4 }}</div>
                </div>
            </div>
        </div>

        <!-- Guest Feedback Mobile -->
        @if($checklist->mod_type === 'room_inspection' && !empty($guestFeedbacks))
        <div class="mobile-section">
            <div class="mobile-section-title">
                <i class="fas fa-star" style="color: #c7a03f;"></i> Guest Feedback
            </div>
            @foreach($guestFeedbacks as $feedback)
            <div class="mobile-feedback-card">
                <div style="display: flex; justify-content: space-between; margin-bottom: 12px;">
                    <span style="font-weight: 700;">{{ $feedback['guest_name'] }}</span>
                    <span style="background: #e5edf2; padding: 2px 12px; border-radius: 30px; font-size: 0.7rem;">#{{ $feedback['room'] }}</span>
                </div>
                <div style="font-size: 0.8rem; color: #5e6f7e; margin-bottom: 12px;">
                    {{ $feedback['arrival'] }} - {{ $feedback['departure'] }}
                </div>
                <div style="font-style: italic; color: #1e405b; border-left: 3px solid #c7a03f; padding-left: 12px;">
                    {{ $feedback['feedback'] }}
                </div>
            </div>
            @endforeach
        </div>
        @endif

        <!-- Public Areas Mobile -->
        <div class="mobile-section">
            <div class="mobile-section-title">
                <i class="fas fa-building"></i> Public Areas
            </div>
            <div style="background: #f9fcff; border-radius: 16px; padding: 16px;">
                @foreach($publicAreas as $area)
                <div style="display: flex; justify-content: space-between; padding: 10px 0; border-bottom: 1px solid #ecf1f5;">
                    <span style="font-size: 0.9rem;">{{ $area['area'] }}</span>
                    <span class="{{ $area['status'] === 'OK' ? 'status-ok' : 'status-issue' }}" style="font-size: 0.75rem;">
                        {{ $area['status'] }}
                    </span>
                </div>
                @endforeach
            </div>
        </div>

        <!-- Room Inspection Mobile -->
        @if($checklist->mod_type === 'room_inspection' && !empty($inspectedRooms))
        <div class="mobile-section">
            <div class="mobile-section-title">
                <i class="fas fa-bed"></i> Rooms Inspected
            </div>
            <div style="display: flex; flex-wrap: wrap; gap: 8px; margin-bottom: 16px;">
                @foreach($inspectedRooms as $room)
                <span style="background: #eef2f6; padding: 6px 16px; border-radius: 30px; font-size: 0.8rem;">Room {{ $room }}</span>
                @endforeach
            </div>
        </div>
        @endif

        <!-- Employee Recognition Mobile -->
        <div class="mobile-section">
            <div class="mobile-section-title">
                <i class="fas fa-award"></i> Star of the Shift
            </div>
            <div style="background: #fdf9e6; border-radius: 16px; padding: 20px; border-left: 6px solid #c7a03f;">
                <div style="font-weight: 700; font-size: 1.1rem;">Putu Ayu</div>
                <div style="font-size: 0.8rem; color: #b45309; margin-bottom: 12px;">Front Office · GSA</div>
                <div style="font-size: 0.9rem; color: #434343;">Assisted VIP early check-in, arranged connecting rooms, welcome amenities. Guest praised her service.</div>
            </div>
        </div>

        <!-- Signature Mobile -->
        <div class="mobile-section" style="border-bottom: none;">
            <div class="mobile-section-title">
                <i class="fas fa-pen"></i> Sign-off
            </div>
            <div class="mobile-signature">
                <div class="mobile-signature-row">
                    <span style="color: #5e6f7e;">Prepared by</span>
                    <span style="font-weight: 600;">{{ $user->name }}</span>
                </div>
                <div class="mobile-signature-row">
                    <span style="color: #5e6f7e;">Date</span>
                    <span>{{ $today->format('d M Y') }}</span>
                </div>
                <div class="mobile-signature-row">
                    <span style="color: #5e6f7e;">Time</span>
                    <span>{{ $today->format('h:i A') }}</span>
                </div>
            </div>
        </div>

        <!-- Back to Dashboard -->
        <div style="padding: 24px 20px 32px; text-align: center;" class="no-print">
            <a href="{{ route('user.dashboard') }}" style="display: inline-block; padding: 14px 28px; background: #0a1c2f; color: white; border-radius: 40px; font-weight: 600; text-decoration: none;">
                <i class="fas fa-arrow-left"></i> Back to Dashboard
            </a>
        </div>
    </div>
</div>
@endif
@endsection

@push('scripts')
<script>
    // Auto-fill signature preview if needed
    document.addEventListener('DOMContentLoaded', function() {
        // Nothing needed for now
    });
</script>
@endpush