<?php

namespace App\Models;

use App\Models\Section;
use App\Models\SubQuestion;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class Question extends Model
{
    use HasFactory;

    /**
     * The table associated with the model.
     *
     * @var string
     */
    protected $table = 'questions';

    /**
     * The attributes that are mass assignable.
     *
     * @var array<string>
     */
    protected $fillable = [
        'checklist_section_id',
        'question_text',
        'type',
        'is_required',
        'has_comment',
        'comment_required',
        'image_required_if_no',
        'order',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'is_required' => 'boolean',
        'has_comment' => 'boolean',
        'comment_required' => 'boolean',
        'image_required_if_no' => 'boolean',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
    ];

    /**
     * The attributes that should be mutated to dates.
     *
     * @var array
     */
    protected $dates = [
        'created_at',
        'updated_at',
    ];

    /**
     * Available question types.
     */
    public const TYPES = [
        'yesNo' => 'Yes/No',
        'header' => 'Header',
        'readOnlyText' => 'Read-Only Text',
        'subQuestionList' => 'Sub-Question List',
        'pageNote' => 'Page Note',
        'imageUpload' => 'Image Upload',
    ];

    /**
     * Relationship: Question belongs to a Section.
     */
    public function section(): BelongsTo
    {
        return $this->belongsTo(Section::class, 'checklist_section_id');
    }

    public function subQuestions()
    {
        return $this->hasMany(SubQuestion::class);
    }

    /**
     * Accessor: Get human-readable type label.
     */
    public function getTypeLabelAttribute(): string
    {
        return self::TYPES[$this->type] ?? 'Unknown';
    }

    /**
     * Accessor: Untuk Image Upload, parse question_text jika JSON
     */
    public function getParsedQuestionTextAttribute()
    {
        if ($this->type === 'imageUpload') {
            // Coba decode JSON
            $decoded = json_decode($this->question_text, true);

            // Jika berhasil decode dan ada original_text, return itu
            if (is_array($decoded) && isset($decoded['original_text'])) {
                return $decoded['original_text'];
            }
        }
        return $this->question_text;
    }

    /**
     * Accessor: Untuk Image Upload, get upload config
     */
    public function getUploadConfigAttribute()
    {
        if ($this->type === 'imageUpload') {
            // Coba decode JSON
            $decoded = json_decode($this->question_text, true);

            if (is_array($decoded)) {
                return [
                    'allowed_formats' => $decoded['allowed_formats'] ?? ['jpg', 'jpeg', 'png'],
                    'max_size_mb' => $decoded['max_size_mb'] ?? 5,
                    'original_text' => $decoded['original_text'] ?? $this->question_text
                ];
            }
        }
        return null;
    }

    /**
     * Scope: Only required questions.
     */
    public function scopeRequired($query)
    {
        return $query->where('is_required', true);
    }

    /**
     * Scope: Only questions that require comment when "No".
     */
    public function scopeNeedsCommentOnNo($query)
    {
        return $query->where('type', 'yesNo')
                     ->where('has_comment', true)
                     ->where('comment_required', true);
    }

    /**
     * Scope: Only questions that require image when "No".
     */
    public function scopeNeedsImageOnNo($query)
    {
        return $query->where('type', 'yesNo')
                     ->where('image_required_if_no', true);
    }

    /**
     * Determine if this question supports comments.
     */
    public function supportsComments(): bool
    {
        return in_array($this->type, ['yesNo', 'header', 'pageNote', 'imageUpload']);
    }

    /**
     * Determine if this question is actionable (can be answered).
     */
    public function isActionable(): bool
    {
        return !in_array($this->type, ['header', 'readOnlyText']);
    }

    /**
     * Untuk SubQuestionList: check if requires image when NO
     */
    public function requiresImageWhenNoForSubQuestions(): bool
    {
        return $this->type === 'subQuestionList' && $this->image_required_if_no;
    }

    /**
     * Mutator: Untuk menyimpan data Image Upload sebagai JSON
     */

    /**
     * Helper: Check if string is JSON
     */
    private function isJson($string)
    {
        if (!is_string($string)) {
            return false;
        }

        json_decode($string);
        return json_last_error() === JSON_ERROR_NONE;
    }

    public function hasSubQuestions(): bool
    {
        return $this->type === 'subQuestionList' && $this->subQuestions()->count() > 0;
    }
}
