<?php
namespace App\Http\Controllers;

use App\Models\ChecklistSubmission;
use App\Models\Cheklist;
use App\Models\Jadwal;
use App\Models\ModSession;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Illuminate\Validation\ValidationException;

class UserController extends Controller
{

    public function __construct()
    {
        Carbon::setLocale('id');
    }

    // UserController.php - method index()
public function index()
{
    $today = Carbon::today();
    $user = Auth::user();

    if (!$this->isEventDate($today)) {
        $modList = [];
        return view('user.dashboard', compact('modList'));
    }

    $modList = $this->buildModList($today, $user);
    
   
    $allModsCompleted = $this->areAllModsCompleted($modList);

    return view('user.dashboard', compact('modList', 'allModsCompleted'));
}

// 🔥 METHOD BARU: Cek apakah semua mod sudah selesai
private function areAllModsCompleted($modList)
{
    if (empty($modList)) {
        return false;
    }

    foreach ($modList as $mod) {
        if ($mod['is_active'] && !$mod['is_completed']) {
            return false; 
        }
    }
    
    return true; 
}

    private function buildModList(Carbon $today, $user = null)
    {
        if (! $user) {
            $user = Auth::user();
        }

        $checklists = Cheklist::orderByRaw("
        CASE
            WHEN hotel = 'HARRIS' THEN 1
            WHEN hotel = 'POP' THEN 2
            ELSE 3
        END
    ")
            ->orderBy('mod_type')
            ->get();

        $modList = [];

        foreach ($checklists as $checklist) {
            // Ambil mod_session spesifik untuk hotel ini
            $modSession = ModSession::where('user_id', $user->id)
                ->where('event_date', $today->toDateString())
                ->where('hotel', $checklist->hotel)
                ->first();

            // Periksa status checklist
            $isCompleted = false;
            $completedAt = null;
            $cardStatus  = 'not_started';

            if ($checklist->mod_type === 'room_inspection') {
                // Room inspection selesai jika sudah 5 kamar
                $roomCount        = $modSession ? $modSession->room_inspection_count : 0;
                $isCompleted      = $roomCount >= 5;
                $generalCompleted = $modSession ? $modSession->general_completed : false;

                // ✅ FIX: Untuk room inspection, hanya selesai jika BOTH general dan room completed
                $allCompleted = $isCompleted && $generalCompleted;

                if ($allCompleted) {
                    $cardStatus  = 'completed';
                    $completedAt = $modSession ? $modSession->updated_at : null;
                } elseif ($roomCount > 0) {
                    $cardStatus = 'in_progress';
                } else {
                    $cardStatus = 'not_started';
                }

                $isCompleted = $allCompleted; // Update isCompleted dengan logika yang benar
            } else {
                // General operation selesai jika ada submission dengan status submitted
                $submission = ChecklistSubmission::where('checklist_id', $checklist->id)
                    ->where('user_id', $user->id)
                    ->whereDate('submitted_at', $today->toDateString())
                    ->where('status', 'submitted')
                    ->first();

                $isCompleted = $submission !== null;

                if ($isCompleted) {
                    $cardStatus  = 'completed';
                    $completedAt = $submission ? $submission->updated_at : null;
                } else {
                    // Cek apakah ada draft/in_progress
                    $draft = ChecklistSubmission::where('checklist_id', $checklist->id)
                        ->where('user_id', $user->id)
                        ->whereDate('submitted_at', $today->toDateString())
                        ->whereIn('status', ['draft', 'in_progress'])
                        ->first();

                    $cardStatus = $draft ? 'in_progress' : 'not_started';
                }
            }

            $modList[] = [
                'hotel'        => $checklist->hotel,
                'mod_type'     => $checklist->mod_type,
                'label'        => $checklist->title,
                'status'       => $cardStatus,
                'completed_at' => $completedAt,
                'checklist_id' => $checklist->id,
                'is_completed' => $isCompleted, // 🔥 FIX: Gunakan $isCompleted yang sudah dihitung
                'is_active'    => true,
            ];
        }

        return $modList;
    }

    private function isEventDate(Carbon $date): bool
    {
        $schedule = Jadwal::where('is_active', true)->first();

        if (! $schedule) {
            return false;
        }

        $events = is_array($schedule->events)
            ? $schedule->events
            : json_decode($schedule->events, true);

        if (! is_array($events)) {
            return false;
        }

        $dateKey = $date->format('Y-m-d');

        return array_key_exists($dateKey, $events);
    }

    public function mod()
    {
        $today      = Carbon::today();
        $isEventDay = $this->isEventDate($today);

        // Ambil checklist dengan urutan: Harris General, Harris Room, Pop General, Pop Room
        $checklists = Cheklist::orderByRaw("
            CASE
                WHEN hotel = 'HARRIS' THEN 1
                WHEN hotel = 'POP' THEN 2
                ELSE 3
            END
        ")->orderByRaw("
            CASE
                WHEN mod_type = 'general' THEN 1
                WHEN mod_type = 'room_inspection' THEN 2
                ELSE 3
            END
        ")->get();

        if (! $isEventDay) {
            return view('user.mod', [
                'accessibleChecklists' => [],
                'allCompleted'         => false,
                'isEventDay'           => false,
                'submitted'            => false,
            ]);
        }

        $user     = Auth::user();
        $todayStr = $today->toDateString();

        $accessibleChecklists = $this->getChecklistAccessibility($checklists, $user, $todayStr);
        $allCompleted = $this->areAllChecklistsCompleted($accessibleChecklists);

        // Cek apakah user sudah submit (signature sudah ada)
        $submitted = ModSession::where('user_id', $user->id)
            ->where('event_date', $today)
            ->whereNotNull('signature_path')
            ->exists();

        return view('user.mod', [
            'accessibleChecklists' => $accessibleChecklists,
            'allCompleted'         => $allCompleted,
            'isEventDay'           => true,
            'submitted'            => $submitted,
        ]);
    }

    /**
     * Proses submit tanda tangan
     */
    public function submitMod(Request $request)
    {
        $request->validate([
            'signature' => 'required|string',
        ]);

        $user  = auth()->user();
        $today = now()->toDateString();

        $sessions = ModSession::where('user_id', $user->id)
            ->where('event_date', $today)
            ->get();

        if ($sessions->isEmpty()) {
            return back()->with('error', 'Tidak ada sesi MOD aktif hari ini.');
        }

        // Pastikan semua checklist sudah selesai
        $allCompleted = true;
        foreach ($sessions as $session) {
            if (!($session->general_completed && $session->room_inspection_completed)) {
                $allCompleted = false;
                break;
            }
        }

        if (!$allCompleted) {
            return back()->with('error', 'Anda harus menyelesaikan semua checklist terlebih dahulu.');
        }

        // Proses simpan signature (base64)
        $signatureData = $request->input('signature');
        $signatureData = preg_replace('/^data:image\/\w+;base64,/', '', $signatureData);
        $signatureData = base64_decode($signatureData);

        if (!$signatureData) {
            return back()->with('error', 'Data tanda tangan tidak valid.');
        }

        $filename = 'signatures/' . $user->id . '_' . $today . '_' . Str::random(8) . '.png';
        Storage::disk('public')->put($filename, $signatureData);

        // Update semua session user hari ini
        foreach ($sessions as $session) {
            $session->update([
                'signature_path' => $filename,
                'submitted_at'   => now(),
            ]);
        }

        return redirect()->route('user.mod')->with('success', 'Tanda tangan berhasil dikirim. MOD Anda sedang menunggu persetujuan admin.');
    }

/**
 * HELPER: Sequential unlocking logic
 */
    private function getChecklistAccessibility($checklists, $user, $today)
    {
        $accessibleChecklists = [];
        $previousCompleted    = true; // Mulai dengan true untuk checklist pertama

        foreach ($checklists as $checklist) {
            $isCompleted = false;
            $roomCount   = 0;

            // Ambil mod_session untuk hotel ini
            $modSession = ModSession::where('user_id', $user->id)
                ->where('event_date', $today)
                ->where('hotel', $checklist->hotel)
                ->first();

            if ($checklist->mod_type === 'room_inspection') {
                // Room inspection selesai jika sudah 5 kamar
                $roomCount   = $modSession ? $modSession->room_inspection_count : 0;
                $isCompleted = $roomCount >= 5;
            } else {
                // General selesai jika ada submission dengan status submitted
                $submission = ChecklistSubmission::where('checklist_id', $checklist->id)
                    ->where('user_id', $user->id)
                    ->whereDate('submitted_at', $today)
                    ->where('status', 'submitted')
                    ->first();

                $isCompleted = $submission !== null;

                // Update mod_session jika general selesai
                if ($isCompleted && $modSession && ! $modSession->general_completed) {
                    $modSession->update(['general_completed' => true]);
                }
            }

            // Checklist dapat diakses jika:
            // 1. Checklist sebelumnya sudah selesai
            // 2. Checklist ini belum selesai
            $canAccess = $previousCompleted && ! $isCompleted;

            $accessibleChecklists[] = [
                'checklist'  => $checklist,
                'can_access' => $canAccess,
                'completed'  => $isCompleted,
                'room_count' => $roomCount,
            ];

            // Update status untuk checklist berikutnya
            if (! $isCompleted) {
                $previousCompleted = false;
            }
        }

        return $accessibleChecklists;
    }

/**
 * HELPER: Cek apakah semua checklist selesai
 */
    private function areAllChecklistsCompleted($accessibleChecklists)
    {
        foreach ($accessibleChecklists as $item) {
            if (! $item['completed']) {
                return false;
            }
        }
        return true;
    }

    public function jadwal()
    {
        $monthNames = [
            1  => 'Januari',
            2  => 'Februari',
            3  => 'Maret',
            4  => 'April',
            5  => 'Mei',
            6  => 'Juni',
            7  => 'Juli',
            8  => 'Agustus',
            9  => 'September',
            10 => 'Oktober',
            11 => 'November',
            12 => 'Desember',
        ];

        $currentMonth = $monthNames[now()->month];
        $currentYear  = now()->year;

        // Ambil jadwal dari database untuk bulan ini
        $jadwal = Jadwal::where('is_active', true)
            ->where('month', $currentMonth)
            ->where('year', $currentYear)
            ->orderBy('created_at', 'desc')
            ->first();

        return view('user.jadwal', compact('jadwal', 'currentMonth', 'currentYear'));
    }

    public function profile()
    {
        return view('user.profile');
    }

    public function updateUsername(Request $request)
    {
        try {
            $request->validate([
                'name' => ['required', 'string', 'max:50', 'unique:users,name,' . Auth::id()],
            ]);

            Auth::user()->update(['name' => $request->name]);

            return back()->with('success', 'Username Berhasil Diupdate');
        } catch (ValidationException $e) {
            return back()->withErrors($e->errors(), 'name')->withInput();
        }
    }

    public function updatePassword(Request $request)
    {
        try {
            $request->validate([
                'current_password' => ['required', 'string'],
                'password'         => ['required', 'string', 'confirmed', 'min:8'],
            ]);

            if (! Hash::check($request->current_password, Auth::user()->password)) {
                throw ValidationException::withMessages([
                    'current_password' => 'Password lama yang Anda masukkan salah.',
                ]);
            }

            Auth::user()->update(['password' => Hash::make($request->password)]);

            return back()->with('success', 'Password Berhasil Diupdate');
        } catch (ValidationException $e) {
            return back()->withErrors($e->errors(), 'password')->withInput();
        }
    }

    public function updateEmail(Request $request)
    {
        try {
            $request->validate([
                'email'    => ['required', 'email', 'max:255', 'unique:users,email,' . Auth::id()],
                'password' => ['required', 'string'],
            ]);

            if (! Hash::check($request->password, Auth::user()->password)) {
                throw ValidationException::withMessages([
                    'password' => 'Password lama yang Anda masukkan salah.',
                ]);
            }

            Auth::user()->update(['email' => $request->email]);

            return back()->with('success', 'Email Berhasil Diupdate');
        } catch (ValidationException $e) {
            return back()->withErrors($e->errors(), 'email')->withInput();
        }
    }
}
