<?php
namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Models\ChecklistSubmission;
use App\Models\Jadwal;
use App\Models\ModSession;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;

class JadwalController extends Controller
{

    private function cleanupModByDates(array $dates)
    {
        foreach ($dates as $date) {
            Log::info("CLEANUP MOD DATE: " . $date);

            $sessions = ModSession::whereDate('event_date', $date)->get();

            foreach ($sessions as $session) {
                Log::info("DELETE SESSION ID: " . $session->id);
                ChecklistSubmission::where('mod_session_id', $session->id)->delete();
                $session->delete();
            }
        }
    }

    /**
     * Save a new schedule with image
     */
    public function store(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'title'  => 'required|string|max:255',
                'month'  => 'required|string|max:50',
                'year'   => 'required|integer|min:2000|max:2100',
                'events' => 'required|json',
                'image'  => 'required|string',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors'  => $validator->errors(),
                ], 422);
            }

            $existing = Jadwal::where('month', $request->month)
                ->where('year', $request->year)
                ->first();

            if ($existing) {
                return response()->json([
                    'success' => false,
                    'message' => 'Schedule for ' . $request->month . ' ' . $request->year . ' already exists',
                ], 409);
            }

            $imageData = $request->image;

            if (preg_match('/^data:image\/(\w+);base64,/', $imageData, $type)) {
                $imageData = substr($imageData, strpos($imageData, ',') + 1);
            }

            if (! base64_decode($imageData, true)) {
                return response()->json([
                    'success' => false,
                    'message' => 'Invalid image format',
                ], 422);
            }

            $timestamp   = time();
            $slugTitle   = Str::slug($request->title);
            $filename    = "schedule_mod_{$slugTitle}_{$timestamp}.png";
            $storagePath = "schedules/{$filename}";

            Storage::disk('public')->put($storagePath, base64_decode($imageData));

            $title = $request->title;

            $jadwal = Jadwal::create([
                'title'      => $title,
                'month'      => $request->month,
                'year'       => $request->year,
                'events'     => json_decode($request->events, true),
                'image_path' => $storagePath,
                'is_active'  => true,
            ]);

            Jadwal::where('id', '!=', $jadwal->id)
                ->update(['is_active' => false]);

            $imageUrl = asset('storage/' . $storagePath);

            return response()->json([
                'success' => true,
                'message' => 'Schedule saved to server successfully',
                'data'    => [
                    'id'         => $jadwal->id,
                    'title'      => $jadwal->title,
                    'month'      => $jadwal->month,
                    'year'       => $jadwal->year,
                    'image_url'  => $imageUrl,
                    'image_path' => $jadwal->image_path,
                    'created_at' => $jadwal->created_at->format('d M Y H:i'),
                ],
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Server error: ' . $e->getMessage(),
            ], 500);
        }
    }

    public function update(Request $request, $id)
    {
        try {
            $validator = Validator::make($request->all(), [
                'title'  => 'required|string|max:255',
                'month'  => 'required|string|max:50',
                'year'   => 'required|integer|min:2000|max:2100',
                'events' => 'required|json',
                'image'  => 'required|string',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors'  => $validator->errors(),
                ], 422);
            }

            $jadwal = Jadwal::findOrFail($id);

            $oldEvents = is_array($jadwal->events)
                ? $jadwal->events
                : json_decode($jadwal->events, true) ?? [];

            $imageData = $request->image;

            if (preg_match('/^data:image\/(\w+);base64,/', $imageData, $type)) {
                $imageData = substr($imageData, strpos($imageData, ',') + 1);
            }

            if (! base64_decode($imageData, true)) {
                return response()->json([
                    'success' => false,
                    'message' => 'Invalid image format',
                ], 422);
            }

            if ($jadwal->image_path && Storage::disk('public')->exists($jadwal->image_path)) {
                Storage::disk('public')->delete($jadwal->image_path);
            }

            $timestamp   = time();
            $slugTitle   = Str::slug($request->title);
            $filename    = "schedule_mod_{$slugTitle}_{$timestamp}.png";
            $storagePath = "schedules/{$filename}";

            Storage::disk('public')->put($storagePath, base64_decode($imageData));

            // Ensure title has "MOD Schedule " prefix if not present (optional)
            $title = $request->title;
            if (! str_starts_with($title, 'MOD Schedule ')) {
                $title = 'MOD Schedule ' . $title;
            }

            $jadwal->update([
                'title'      => $title,
                'month'      => $request->month,
                'year'       => $request->year,
                'events'     => json_decode($request->events, true),
                'image_path' => $storagePath,
                'is_active'  => true,
            ]);

            Jadwal::where('id', '!=', $jadwal->id)
                ->update(['is_active' => false]);

            $newEvents = json_decode($request->events, true) ?? [];

            $deletedDates = array_diff(array_keys($oldEvents), array_keys($newEvents));

            Log::info("DELETED DATES: " . json_encode($deletedDates));

            if (! empty($deletedDates)) {
                $this->cleanupModByDates($deletedDates);
            }

            $imageUrl = asset('storage/' . $storagePath);

            return response()->json([
                'success' => true,
                'message' => 'Schedule updated successfully',
                'data'    => [
                    'id'         => $jadwal->id,
                    'title'      => $jadwal->title,
                    'month'      => $jadwal->month,
                    'year'       => $jadwal->year,
                    'image_url'  => $imageUrl,
                    'image_path' => $jadwal->image_path,
                    'created_at' => $jadwal->updated_at->format('d M Y H:i'),
                ],
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Server error: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Get all schedules
     */
    public function index()
    {
        try {
            $schedules = Jadwal::orderBy('year', 'desc')
                ->orderBy('created_at', 'desc')
                ->get()
                ->map(function ($schedule) {
                    return [
                        'id'             => $schedule->id,
                        'title'          => $schedule->title,
                        'month'          => $schedule->month,
                        'year'           => $schedule->year,
                        'events_count'   => $schedule->events ? count($schedule->events) : 0,
                        'image_url'      => $schedule->image_path ? asset('storage/' . $schedule->image_path) : null,
                        'image_path'     => $schedule->image_path,
                        'created_at'     => $schedule->created_at->format('d M Y H:i'),
                        'is_from_server' => true,
                    ];
                });

            return response()->json([
                'success' => true,
                'data'    => $schedules,
                'total'   => $schedules->count(),
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to load schedules from server',
            ], 500);
        }
    }

    /**
     * Show schedule details
     */
    public function show($id)
    {
        try {
            $schedule = Jadwal::findOrFail($id);

            return response()->json([
                'success' => true,
                'data'    => [
                    'id'         => $schedule->id,
                    'title'      => $schedule->title,
                    'month'      => $schedule->month,
                    'year'       => $schedule->year,
                    'events'     => $schedule->events,
                    'image_url'  => $schedule->image_path ? asset('storage/' . $schedule->image_path) : null,
                    'image_path' => $schedule->image_path,
                    'created_at' => $schedule->created_at->format('d M Y H:i'),
                ],
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Schedule not found',
            ], 404);
        }
    }

    /**
     * Delete schedule
     */
    public function destroy($id)
    {
        try {
            $schedule = Jadwal::findOrFail($id);
            $title    = $schedule->title;

            if ($schedule->image_path && Storage::disk('public')->exists($schedule->image_path)) {
                Storage::disk('public')->delete($schedule->image_path);
            }

            $schedule->delete();
            $events = is_array($schedule->events)
                ? $schedule->events
                : json_decode($schedule->events, true) ?? [];

            $dates = array_keys($events);

            Log::info("DELETE SCHEDULE, CLEAN DATES: " . json_encode($dates));

            if (! empty($dates)) {
                $this->cleanupModByDates($dates);
            }

            return response()->json([
                'success' => true,
                'message' => "Schedule '{$title}' deleted from server successfully",
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete schedule: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Backup method for compatibility
     */
    public function uploadScheduleImage(Request $request)
    {
        return $this->store($request);
    }

    /**
     * Get list of schedule images
     */
    public function getScheduleImages()
    {
        try {
            $schedules = Jadwal::whereNotNull('image_path')
                ->orderBy('created_at', 'desc')
                ->get();

            $images = $schedules->map(function ($schedule) {
                return [
                    'id'                   => $schedule->id,
                    'name'                 => basename($schedule->image_path),
                    'url'                  => asset('storage/' . $schedule->image_path),
                    'title'                => $schedule->title,
                    'month'                => $schedule->month,
                    'year'                 => $schedule->year,
                    'created_at'           => $schedule->created_at->timestamp,
                    'created_at_formatted' => $schedule->created_at->format('d M Y H:i'),
                ];
            });

            return response()->json([
                'success' => true,
                'data'    => $images,
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to load schedule images',
            ], 500);
        }
    }

    /**
     * Delete schedule image
     */
    public function deleteScheduleImage(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'filename' => 'required|string',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Invalid filename',
                ], 422);
            }

            $schedule = Jadwal::where('image_path', 'like', '%' . $request->filename . '%')->first();

            if (! $schedule) {
                return response()->json([
                    'success' => false,
                    'message' => 'Schedule not found',
                ], 404);
            }

            if ($schedule->image_path && Storage::disk('public')->exists($schedule->image_path)) {
                Storage::disk('public')->delete($schedule->image_path);
            }

            $schedule->update(['image_path' => null]);

            return response()->json([
                'success' => true,
                'message' => 'Schedule image deleted successfully',
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete schedule image',
            ], 500);
        }
    }
}