<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Models\AdminRequest;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\ValidationException;

class AuthController extends Controller
{
    public function showLogin()
    {
        return view('login');
    }

    public function pending()
    {
        return view('adminpending');
    }

    public function login(Request $request)
{
    $request->validate([
        'email'    => 'required|email',
        'password' => 'required',
    ]);

    $user = User::where('email', $request->email)->first();

    // Cek kredensial
    if (!$user || !Hash::check($request->password, $user->password)) {
        throw ValidationException::withMessages([
            'email' => 'Email atau password salah.',
        ]);
    }

    // Cek apakah akun sudah disetujui admin
    if ($user->status_mod !== 'approved') {
        throw ValidationException::withMessages([
            'email' => 'Akun Anda masih menunggu persetujuan admin.',
        ]);
    }

    // Jika tidak pakai TOTP, langsung login
    Auth::login($user, $request->boolean('remember'));
    $request->session()->regenerate();

    // Redirect berdasarkan role
    return $user->role_id == 1
        ? redirect()->intended('/admin/dashboard')
        : redirect()->intended('/user/dashboard');
}

    public function register(Request $request)
    {
        $request->validate([
            'name'     => 'required|string|max:255',
            'email'    => 'required|email|unique:users',
            'password' => 'required|min:6|confirmed',
            'role'     => 'required|in:User,Admin',
        ]);

        $roleId = $request->role === 'Admin' ? 1 : 2;
        $statusMod = $roleId == 1 ? 'pending' : 'approved';

        $user = User::create([
            'name'       => $request->name,
            'email'      => $request->email,
            'password'   => Hash::make($request->password),
            'role_id'    => $roleId,
            'status_mod' => $statusMod
        ]);

        if ($roleId == 1) {
            AdminRequest::create([
                'user_id'           => $user->id,
                'requested_role_id' => 1,
                'status'            => 'pending',
            ]);
        }

        Auth::login($user);

        if ($user->status_mod === 'pending') {
            return redirect()->route('adminpending')
                ->with('success', 'Akun admin berhasil dibuat. Menunggu persetujuan.');
        }

        return redirect('/user/dashboard')
            ->with('success', 'Registrasi berhasil. Selamat datang!');
    }

    public function logout(Request $request)
    {
        Auth::logout();
        $request->session()->invalidate();
        $request->session()->regenerateToken();

        return redirect('/login');
    }
}