<?php
namespace App\Http\Controllers;

use Carbon\Carbon;
use App\Models\User;
use App\Models\Jadwal;
use App\Models\Cheklist;
use App\Models\ModSession;
use App\Models\AdminRequest;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Http\RedirectResponse;
use Illuminate\Validation\ValidationException;

class AdminController extends Controller
{
    public function index()
    {
            $pendingUsersCount = AdminRequest::where('status', 'pending')->count();
            $today = now()->toDateString();
        
            // Total users (role_id = 2)
            $totalUsers = User::where('role_id', 2)->count();

            // Recent activities: ambil 5 admin request terbaru yang masih memiliki user
        $recentActivities = AdminRequest::with('user')
        ->whereHas('user')  // Hanya ambil yang user-nya masih ada
        ->latest()
        ->take(5)
        ->get()
        ->map(function ($req) {
            return [
                'icon'   => $req->status == 'pending' ? 'clock' : ($req->status == 'approved' ? 'check-circle' : 'times-circle'),
                'title'  => $req->user->name . ' mengajukan request admin',
                'time'   => $req->created_at->diffForHumans(),
                'status' => $req->status,
            ];
        });
        
        // Jika kurang dari 5, tambahkan dari ModSession yang baru submit (juga dengan filter user valid)
        if ($recentActivities->count() < 5) {
        $recentSubmissions = ModSession::with('user')
            ->whereHas('user')
            ->whereNotNull('signature_path')
            ->latest('updated_at')
            ->take(5 - $recentActivities->count())
            ->get()
            ->map(function ($session) {
                return [
                    'icon'   => 'clipboard-check',
                    'title'  => $session->user->name . ' menyelesaikan MOD',
                    'time'   => $session->updated_at->diffForHumans(),
                    'status' => 'submitted',
                ];
            });
        $recentActivities = $recentActivities->concat($recentSubmissions)->sortByDesc('time')->take(5);
        }
        
            // Upcoming jadwal: ambil jadwal aktif, urut berdasarkan tahun dan bulan terbaru
        $upcomingJadwal = Jadwal::where('is_active', true)
        ->orderBy('year', 'desc')
        ->orderByRaw("FIELD(month, 'Januari','Februari','Maret','April','Mei','Juni','Juli','Agustus','September','Oktober','November','Desember')")
        ->take(5)
        ->get()
        ->map(function ($jadwal) {
            $events = $jadwal->events ?? [];
            $firstDate = null;
            if (is_array($events) && !empty($events)) {
                $dates = array_keys($events);
                sort($dates);
                $firstDate = $dates[0] ?? null;
            }
            return [
                'name' => $jadwal->title,
                'date' => $firstDate ? Carbon::parse($firstDate)->format('d M Y') : $jadwal->month . ' ' . $jadwal->year,
                'time' => '08:00 - 17:00', // bisa disesuaikan
            ];
        });
        
            return view('admin.dashboard', compact(
                'totalUsers',
                'recentActivities',
                'upcomingJadwal',
                'pendingUsersCount',
            ));
        }

    public function indexman()
    {
        return view('admin.manajemen');
    }

    public function indexjad()
    {
        return view('admin.jadwal');
    }

    public function indexset()
    {
        return view('admin.settings');
    }

    // Di dalam AdminController.php

    public function updateUsername(Request $request)
    {
        try {
            $request->validate([
                'name' => ['required', 'string', 'max:50', 'unique:users,name,' . Auth::id()],
            ]);

            Auth::user()->update(['name' => $request->name]);

            return back()->with('success', 'Username Berhasil Diupdate');
        } catch (ValidationException $e) {
            return back()->withErrors($e->errors(), 'name')->withInput();
        }
    }

    public function updatePassword(Request $request)
    {
        try {
            $request->validate([
                'current_password' => ['required', 'string'],
                'password'         => ['required', 'string', 'confirmed', 'min:8'],
            ]);

            if (! Hash::check($request->current_password, Auth::user()->password)) {
                throw ValidationException::withMessages([
                    'current_password' => 'Password lama yang Anda masukkan salah.',
                ]);
            }

            Auth::user()->update(['password' => Hash::make($request->password)]);

            return back()->with('success', 'Password Berhasil Diupdate');
        } catch (ValidationException $e) {
            return back()->withErrors($e->errors(), 'password')->withInput();
        }
    }

    public function updateEmail(Request $request)
    {
        try {
            $request->validate([
                'email'    => ['required', 'email', 'max:255', 'unique:users,email,' . Auth::id()],
                'password' => ['required', 'string'],
            ]);

            if (! Hash::check($request->password, Auth::user()->password)) {
                throw ValidationException::withMessages([
                    'password' => 'Password lama yang Anda masukkan salah.',
                ]);
            }

            Auth::user()->update(['email' => $request->email]);

            return back()->with('success', 'Email Berhasil Diupdate');
        } catch (ValidationException $e) {
            return back()->withErrors($e->errors(), 'email')->withInput();
        }
    }

    public function request()
    {
        $search = request('search', '');
        $sort   = request('sort', 'name_asc');

        $query = AdminRequest::with('user')
            ->where('status', 'pending')
            ->whereHas('user');

        if ($search) {
            $query->whereHas('user', function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                    ->orWhere('email', 'like', "%{$search}%");
            });
        }

        $query->join('users', 'admin_requests.user_id', '=', 'users.id');
        if ($sort === 'name_asc') {
            $query->orderBy('users.name', 'asc');
        } elseif ($sort === 'name_desc') {
            $query->orderBy('users.name', 'desc');
        } else {
            $query->latest('admin_requests.created_at');
        }

        $requests = $query->select('admin_requests.*')->paginate(10);

        return view('admin.adminrequest', compact('requests'));
    }

    public function status()
{
    $today = now()->toDateString();

    $users = User::where('role_id', 2)->get();
    $userStatuses = [];

    foreach ($users as $user) {

        $sessions = ModSession::where('user_id', $user->id)
            ->where('event_date', $today)
            ->get();

        $hasSession = $sessions->count() > 0;

        $allCompleted = $sessions->count() > 0 &&
            $sessions->every(fn ($s) =>
                $s->general_completed &&
                $s->room_inspection_completed &&
                $s->status === 'completed'
            );

        $hasSignature = $sessions->whereNotNull('signature_path')->count() > 0;

        if (! $hasSession) {
            $statusLabel = 'Belum Mengisi MOD';
            $badgeClass  = 'danger';
            $icon        = 'clock';

        } elseif ($allCompleted && ! $hasSignature) {
            $statusLabel = 'Menunggu Tanda Tangan';
            $badgeClass  = 'warning';
            $icon        = 'pencil-alt';

        } else {
            $statusLabel = 'Menunggu Persetujuan';
            $badgeClass  = 'info';
            $icon        = 'hourglass-half';
        }

        $userStatuses[] = [
            'user'         => $user,
            'status_label' => $statusLabel,
            'badge_class'  => $badgeClass,
            'icon'         => $icon,
        ];
    }

    return view('admin.usermod', [
        'users' => $userStatuses,
        'date'  => now()->format('d F Y'),
        'total' => count($userStatuses),
    ]);
}

    public function approve($id)
    {
        $request = AdminRequest::findOrFail($id);

        $user = User::findOrFail($request->user_id);

        $user->update([
            'role_id'    => 1,
            'status_mod' => 'approved',
        ]);

        $request->update([
            'status' => 'approved',
        ]);

        return back()->with('success', 'Permintaan admin disetujui');
    }

    public function reject($id)
    {
        $request = AdminRequest::findOrFail($id);
        $request->delete();

        return back()->with('success', 'Permintaan admin ditolak');
    }

    public function userman()
    {
        $sort   = request('sort', 'name_asc');
        $search = request('search');

        $query = User::where('role_id', 2);

        if ($search) {
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                    ->orWhere('email', 'like', "%{$search}%");
            });
        }

        if ($sort === 'name_asc') {
            $query->orderBy('name', 'asc');
        } elseif ($sort === 'name_desc') {
            $query->orderBy('name', 'desc');
        } else {
            $query->latest();
        }

        $users = $query->paginate(10); // atau ->get() jika tanpa pagination

        return view('admin.manajemen.usermanajemen', compact('users', 'search'));
    }

    public function userdestroy(User $user): RedirectResponse
    {
        if ($user->role_id !== 2) {
            return redirect()->back()->with('error', 'Anda tidak dapat menghapus user ini.');
        }

        // Hapus user
        $userName = $user->name;
        $user->delete();

        return redirect()->back()->with('success', "User $userName berhasil dihapus.");
    }
}
